
% IRIS SEGMENTATION BASED ON THE GENERALIZED STRUCTURE TENSOR (GST)
% (C) Fernando Alonso-Fernandez, Josef Bigun
% Halmstad University, Sweden - http://islab.hh.se/
% Contact: feralo@hh.se
% Date of current release: October 2019 
% (please check the URL above for more recent releases)
%Compiler: Matlab r2018b 64 bits (mcc command) under Windows 10

%References used in the comments below. 
%PLEASE CITE THESE REFERENCES IF YOU PUBLISH ANY WORK MAKING USE OF THIS CODE
%[1]
%F. Alonso-Fernandez, J. Bigun, Iris Boundaries Segmentation Using the
%Generalized Structure Tensor. A Study on the Effects on Image Degradation, 
%Proc. Intl Conf on Biometrics: Theory, Apps and Systems, BTAS, 
%Washington DC, September 23-26, 2012

%[2] 
%F. Alonso-Fernandez, J. Bigun, Near-Infrared and Visible Light Periocular
%Recognition with Gabor Features using Frequency-Adaptive Automatic Eye
%Detection, IET Biometrics, 4 (2): 74-89, June 2015, ISSN: 2047-4938


%This script is the main segmentation function
%You should modify it appropriately to run on your desired database
%(instructions are given below)

%A rough description of the steps carried out by the code is as follows 
%(some steps can be deactivated as indicated later):

%1-Image downsampling for speed purposes. 
%This will not jeopardize accuracy, since the detected iris circles are
%later fitted to the irregular iris contours, so any loss of resolution in
%iris circles detection due to downsampling is compensated

%2-Contrast normalization based on imadjust (Matlab function)
%This increases the image contrast, spreading grey values fully in the
%0-255 range

%3-Specular reflection removal based on the method published in:
%C. Rathgeb, A. Uhl, P. Wild, "Iris Biometrics. From Segmentation to 
%Template Security", Springer, 2013

%4-Computation of the image frequency based on the method published in [2]
%This helps to customize inner parameters of steps 5-8 below to the
%input image

%5-Adaptive eyelash removal using the image frequency, as indicated in [2]
%Method based on p-rank order filters as published in:
%He, Z., Tan, T., Sun, Z., Qiu, X.: "Toward accurate and fast iris
%segmentation for iris biometrics", IEEE Trans. Pattern Anal. Mach.
%Intell., 2010, 31, (9), pp. 12951307
%Eyelashes create strong edges that may mislead the filters used for eye 
%center estimation and iris segmentation in steps 7 and 9

%6-Adaptive edge map computation using the image frequency [2]
%Edge map is the basis for eye center estimation and iris boundaries
%detection, see refences [1, 2] for further details

%7-Estimation of eye center based on the method published in [2] using
%circular symmetry filters. 
%The estimated center is used to mask candidate regions for the centers of 
%iris circles, helping to improve detection accuracy

%8-Detection of eyelids based on linear symmetry detection 
%(unpublished and unoptimized, only return a straight line). 
%The detected eyelids are used to mask candidate regions for the centers 
%of iris circles too, helping to improve detection accuracy

%9-Detection of iris boundaries based on the method published in [1]
%In NIR images, the inner (pupil) circle is detected first
%In VIS images, the outer (sclera) filter is detected first

%10-Irregular contour fitting based on active contours as published in:
%J. Daugman. New methods in iris recognition. IEEE TSMC-B, 37(5), 2007

clear

%%%%RANGE OF PUPIL AND SCLERA RADII
%%%%You should define a range of expected radii for your database%%%%

%Expected range of inner (pupil) circle (define as appropriate for your data)
rmin_pupil=18;
rmax_pupil=80;
%Expected range of outer (sclera) circle (define as appropriate for your data)
rmin_sclera=90;
rmax_sclera=135;

%%%%Below are some ranges for the indicated databases, extracted from
%%%%groundtruth data, as published in:

%F. Alonso-Fernandez, J. Bigun, Near-Infrared and Visible Light Periocular
%Recognition with Gabor Features using Frequency-Adaptive Automatic Eye
%Detection, IET Biometrics, 4 (2): 74-89, June 2015, ISSN: 2047-4938

%Biosec: 18-80/90-135
%Mobbio: 4-30/18-56
%CASIA Interval: 20-75/75-130
%IITD: 18-75/80-125
%Notredame: 25-85/90-140
%Ubiris v2: 5-50/20-135

%%%%Groundtruth segmentation data of these databases is available online,
%%%%please check the following URL for more information:

%http://islab.hh.se/mediawiki/Iris_Segmentation_Groundtruth



%%%%CREATION OF CIRCULAR FILTERS FOR IRIS DETECTION
%This creates the circular filters of different radius that will be used
%for iris boundaries detection. They are created only once for speed 
%purposes and read by the segmentation function. Filters should
%fully cover the range of pupil and sclera radii for your database, so the
%function receives as input the upper limits of expected radii set above.
%This function creates a file "h2_filters.mat" in the current folder.
%THIS FILE SHOULD BE AVAILABLE WHEN CALLING TO THE
%FUNCTION iris_segment BELOW, SO IT CAN BE READ BY iris_segment 
%generate_filters(rmax_pupil,rmax_sclera);
dos(['generate_filters ' num2str(rmax_pupil) ' ' num2str(rmax_sclera)])
%Load the variable. It will add the following variables to the workspace:
%'h2_filters.mat','h2_pupil','h2_radio_pupil','h2_sclera','h2_radio_sclera'
%load h2_filters.mat
%h2_rmin=h2_radio_pupil(1);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



%%%%CUSTOMIZABLE INPUT PARAMETERS
%If any of these parameters are not set to a valid value, the
%segmentation function will fix them to the default value

%%%%IMAGE SCALING FOR SPEED PURPOSES
%(default=0.5, allowed values=(0,1])
scale=0.5;
%Resize your images to a smaller size if the execution
%takes too much (e.g. make scale=0.5 to resize at 50%).
%Lower pixel accuracy is compensated with irregular contour fitting

%If scale=1, images are processed at their original size
%A value higher than 1 is not allowed

%Consider which is the lowest value of "rmin_pupil" when resizing,
%since a very low value will make pupil detection
%unreliable. For example, Mobbio or UBIRIS databases have
%images with a small pupil circle, of just 4 or 5 pixels,
%so scaling is not recommended. A value of rmin_pupil
%smaller than 4 is not allowed by the segmentation function
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%STEP IN CIRCULAR FILTERS (default=2, allowed values={1,2})
step=2;
%A circular filter of variable (increasing) radius is used
%for pupil and sclera detection. By default, step=2 is used for
%higher speed (increase of 2 pixels in radius).
%Lower pixel accuracy is compensated with irregular contour fitting
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%DEFINE SENSOR TYPE (uncomment the appropriate)
%In NIR images, the inner (pupil) circle is detected first
%In VIS images, the outer (sclera) filter is detected first
%This is because in NIR images, pupil-to-iris transition is sharper
%than iris-to-sclera transition, thus more reliable to detect first.
%The opposite happens with VIS images
%If not defined correctly, it will be set to 'NIR'
sensor_type='NIR'; %Near-infrared
%sensor_type='VIS'; %Visible
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%IMAGE ENHANCEMENT / PREPROCESSING
%You may want to apply your own enhancement / preprocessing here
%to the iris image before calling the segmentation function

%The segmentation function contains the following preprocessing, which
%is applied in the order specified.

%Deactivate flags accordingly if you want to remove some of the steps

%Contrast normalization based on Matlab imadjust
%(default=1, allowed values={0,1})
contrast_normalization=1; %make 0 to deactivate;

%Specular reflection removal (default=1, allowed values={0,1})
reflection_removal=1; %make 0 to deactivate;

%Eyelash removal (default=1, allowed values={0,1})
eyelash_removal=1; %make 0 to deactivate;

%Strength of eyelash_removal (default=1, allowed values={0,1})
%Higher strength uses a bigger p-rank filter and is more effective in
%case of prominent eyelashes. Only used if eyelash_removal=1
eyelash_strength=1; %make 0 for lower strength;

%Detection of eyelids based on linear symmetry detection
%(default=1, allowed values={0,1})
eyelids_detection=1; %make 0 to deactivate;


%%%%IRREGULAR CONTOUR FITTING (default=1, allowed values={0,1})
%Deactivate if your are only interested in iris circles, or if you want
%to apply your own contour fitting algorithm afterwards
contour_fitting=1; %make 0 to deactivate;


%%%%VERBOSITY (default=0, allowed values={0,1})
verbosity=0; %make 1 to activate



%Here you should make a loop to read your target images and input them to
%the segmentation function

%%%%LOOP SHOULD START HERE (SEE BELOW FOR END)

%for image=1...
    
    %Set path and image name of your input file, and path of output results
    
    %Complete path to the folder with the iris image
    path_in='C:\my_folder';
    
    %Name of iris image in the above path (with extension)
    %Accepted formats: those supported by Matlab "imread" (only uint8)
    %Accepts both RGB and gray images (RGB will be converted to gray)
    image_file='iris_image.bmp'    
    
    %Complete path where you want that the function saves the results
    %The function will use "image_file" (without extension) as basis to
    %store the results, see below for details
    path_out='C:\results'; 
    %path_out='.\' %if you want to use the current folder
    
        
    %%%%SEGMENTATION FUNCTION
%     tic
%     iris_segment(rmin_pupil,rmax_pupil,rmin_sclera,rmax_sclera, ...
%         step, sensor_type,scale,...
%         contrast_normalization,reflection_removal,eyelash_removal,eyelash_strength,eyelids_detection,contour_fitting,verbosity, path_in, image_file, path_out);
%     toc
    
    %Call the segmentation function (leave the following as is)    
    tic
    dos(['iris_segment ' num2str(rmin_pupil) ' ' num2str(rmax_pupil) ' ' num2str(rmin_sclera) ' ' num2str(rmax_sclera) ' ' ...
        num2str(step) ' ' sensor_type ' ' num2str(scale) ' ' ...
        num2str(contrast_normalization) ' ' num2str(reflection_removal) ' ' ...
        num2str(eyelash_removal) ' ' num2str(eyelash_strength) ' ' ...
        num2str(eyelids_detection) ' ' num2str(contour_fitting) ' ' ...
        num2str(verbosity) ' ' path_in ' ' image_file ' ' path_out]);
    toc
    
    
    %%%%THE FUNCTION WRITES THE FOLLOWING RESULTS IN THE INDICATED PATH
        
    %Image with segmentation results (iris circles and eyelids)
    %Circles are in red
    %Irregular fitted contours are in green (if enabled)
    %[path_out image_file '_circles.png'])
    
    %INTERMEDIATE IMAGES DUE TO PREPROCESSING
    %If some preprocessing is disabled, some images will be equal
    %Image after contrast normalization 
    %[path_out image_file '_1_fimadjust.png']
    %Image after specular reflection removal
    %[path_out image_file '_2_fspec.png']
    %Image after eyelash removal
    %[path_out image_file '_3_frank.png']
    
    %Complex edge map
    %[path_out image_file '_I20linear.png']
    
    %Binary segmentation mask
    %[path_out image_file '_mask.png']
    
    %Binary segmentation mask in Matlab format    
    %[path_out image_file '_mask.mat']
    
    %Segmentation data
    %iup/idown: row of upper/lower eyelid
    %rpupil, CC_pupil: pupil circle
    %rsclera, CC_sclera: sclera circle
    %ic, jc: coordinates of pre-detected eye center
    %pupil_contour, sclera_contour: coordinates of irregular (fitted) contours
    %[path_out image_file '_segm_data.mat']
    

    
%end

%%%%LOOP SHOULD END HERE

